#!/usr/bin/python3

import os
import time
import json
import logging
from watchdog.observers import Observer
from watchdog.events import FileSystemEventHandler
from deepdiff import DeepDiff
 
 # Configure logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class MyHandler(FileSystemEventHandler):
    def __init__(self, bacnet_map_file):
        self.last_modified = {}
        self.sensor_map = {}
        with open('/var/config/scada/sensors_map.json', 'r') as file:
            self.sensor_map = json.load(file)
        self.bacnet_map_file = bacnet_map_file

    def on_modified(self, event):
        if event.is_directory:
            return
        elif event.event_type == 'modified' and event.src_path.endswith('sensors_map.json'):
            current_time = time.time()
            file_path = event.src_path
            # Check if the file was modified less than a second ago
            if (current_time - self.last_modified.get(file_path, 0)) > 1:
                logging.info(f'JSON file {file_path} has been modified. Getting modified sensor map')         
                self.check_and_get_modified_entry(file_path)
                #print("Modified Entry:", modified_entry)
                with open(file_path, 'r') as file:
                    self.sensor_map = json.load(file)
                self.last_modified[file_path] = current_time
                
    def check_and_get_modified_entry(self, file_path):
        # Read the JSON data from the input file
        logging.info(f'imput ifle is {file_path}')
        try:
            with open(file_path, 'r') as input_file:
                data = json.load(input_file)
            self.check_sensor_map(data)
        except FileNotFoundError:
            logging.error(f"sensor map nor at {file_path} not found.")    
               
    def check_sensor_map(self, sensor_map):

        diff = DeepDiff(self.sensor_map, sensor_map, ignore_order=True)
        
        
        #print(f"sensors removed {removed_sensor}")
        if "iterable_item_added" in diff:
            added_sensor = diff.get("iterable_item_added", {})
            id_key = next((key for key in added_sensor.keys() if 'id' in added_sensor[key]), '')
            eui = added_sensor.get(id_key, {}).get('id', '')
            eui_without_dashes = eui.replace('-', '')
            eui_last_four = eui_without_dashes[-4:]
            sensor_name = added_sensor.get(id_key, {}).get('sensor', '')
            src = added_sensor.get(id_key, {}).get('src', '')
            logging.info(f"added sensor {eui}, {sensor_name}, {src}")
            sensor_definition = self.get_sensor_definition(sensor_name, src)
            bacnet_objects = self.create_bacnet_objects(eui, eui_last_four, src, sensor_definition)
            self.append_bacnet_map(bacnet_objects)

        if "iterable_item_removed" in diff:
            removed_sensor = diff.get("iterable_item_removed")
            id_key = next((key for key in removed_sensor.keys() if 'id' in removed_sensor[key]), '')
            eui = removed_sensor.get(id_key, {}).get('id', '')
            sensor_name = removed_sensor.get(id_key, {}).get('sensor', '')
            logging.info(f"removed sensor {eui}, {sensor_name}: All objects have been removed.")

    def get_sensor_definition(self, sensor_path, src):
        
        if ("radiobridge" in sensor_path or "elsys" in sensor_path or "adenuis" in sensor_path):
            sensor_definition_path = os.path.join("/etc/scada/sensors", src, f"{sensor_path}.json")
            #print(sensor_definition_path)
            with open(sensor_definition_path, 'r') as definition_file:
                data = json.load(definition_file)
                properties = data.get('properties', {})
                return properties
        else:
            sensor_definition_path = os.path.join("/var/config/scada/sensors", src, f"{sensor_path}.json")
            #logging.info(f"other sensor path: {sensor_definition_path}")
            with open(sensor_definition_path, 'r') as definition_file:
                data = json.load(definition_file)
                properties = data.get('properties', {})
                return properties
            
    def create_bacnet_objects(self, eui, eui_last_four, src, properties):
        existing_oids = self.get_existing_oids()
        new_bacnet_objects = []

        for property, details in properties.items():
            type = details.get("type", "")
            data_type = self.get_type(type)

            new_oid = self.get_new_oid(existing_oids)
            existing_oids.add(new_oid)

            new_bacnet_object = {data_type: {
                "descr": "",
                "key": f"{src}@{eui}@{property}",
                "name": f"{property}-{eui_last_four}",
                "oid": new_oid,
             }}
            new_bacnet_objects.append(new_bacnet_object)

        return new_bacnet_objects

    def get_existing_oids(self):
        # Load existing content from bacnet_object_file
        try:
            with open(self.bacnet_map_file, 'r') as file :
                bacnet_map = json.load(file)
            existing_content = bacnet_map
            existing_oids = set()

            for object_type, objects in existing_content.items():
                if isinstance(objects, list):
                    for obj in objects:
                        oid = obj.get("oid")
                        if oid is not None:
                            existing_oids.add(oid)

            return existing_oids
        except FileNotFoundError:
            logger.warning(f"BACnet map file not found: {self.bacnet_map_file}")
            return set()
    def get_new_oid(self, existing_oids):
        new_oid = 1
        while new_oid in existing_oids:
            new_oid += 1
        return new_oid
      
    def append_bacnet_map(self, bacnet_array):
        with open(self.bacnet_map_file, 'r') as file :
            bacnet_map = json.load(file)
        existing_content = bacnet_map

        for obj in bacnet_array:
            object_type = next(iter(obj))
            new_object = obj[object_type]
            existing_content.setdefault(object_type, []).append(new_object)

        logger.info('Updated Content:')
        logger.info(json.dumps(existing_content, indent=2))
        try:
            with open(self.bacnet_map_file, "w") as file:
                json.dump(existing_content, file, indent=2) 
        except FileNotFoundError:
            logging.error(f"BACnet map file not found: {self.bacnet_map_file}")
    def get_type(self, type):
        if "int" in type or "float" in type:
            return "analog-values"
        if "bool" in type:
            return "binary-values"
        if "string" in type:
            return "character-string-values"
if __name__ == "__main__":

    #input_file_path = '/var/config/scada/sensors_map.json'
    path = '/var/config/scada/'
    bacnet_map_file = "/var/config/scada/bacnet_map.json"
    event_handler = MyHandler(bacnet_map_file)
    observer = Observer()
    observer.schedule(event_handler, path, recursive=False)
   
    print(f'Monitoring directory {path} for changes to JSON files...')
   
    observer.start()
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        observer.stop()
    observer.join()
